#include "hnfsflt.h"

atomic_t hnfsflt_device_major;
static atomic_t device_open = ATOMIC_INIT(0);	/* Indicates if the device is open, only one reader allowed */
static atomic_t in_devread = ATOMIC_INIT(0);	 /* Indicates if there is an ongoing read operation */
wait_queue_head_t read_waitq;	/* queue of read in waiting state */

/* Current logline to return */
static struct hnfsflt_logline *ll_current = NULL;

/*
 * Utility macro to init/check ll_current.
 * - declare 'bytes_read' and set to 0 to block
 * - must be included in a loop
 * - will return on error
 */
#define INIT_LL_CURRENT									\
		if (!ll_current) {							\
			/* No logline in read buffer, get next one from ring buffer */	\
			ll_current = hnfsflt_ll_take(bytes_read == 0);			\
			if (ll_current == NULL) {					\
				break;							\
			}								\
			/* No new logline: reset current value before returning */	\
			if (IS_ERR(ll_current)) {					\
				ssize_t ret = PTR_ERR(ll_current);			\
				ll_current = NULL;					\
				atomic_dec(&in_devread);				\
				return ret;						\
			}								\
		}

/*
 * Called when a process tries to open the device file, like
 * "cat /dev/hnfsflt"
 */
static int hnfsdev_open(struct inode *inode, struct file *file)
{
	if (file->f_mode & FMODE_WRITE)
		return -EPERM;

	if (file->f_flags & O_NONBLOCK)
		return -EINVAL;

	if (atomic_add_return(1, &device_open) > 1) {
		atomic_dec(&device_open);
		return -EBUSY;
	}

	/* we allow to read from the char device after return,
		so don't unload the module until release called */
	try_module_get(THIS_MODULE);

	return 0;
}


/*
 * Called when a process closes the device file.
 */
static int hnfsdev_release(struct inode *inode, struct file *file)
{

	/* free ll_current if not fully read */
	if (ll_current != NULL) {
		hnfsflt_ll_free(ll_current);
		ll_current = NULL;
	}

	atomic_dec(&device_open);

	/*
	 * Decrement the module usage count.
	 */
	module_put(THIS_MODULE);

	return 0;
}


/*
 * Called when a process, which already opened the dev file, attempts to
 * read from it.
 * We make sure we have only one reader in open/release.
 */
static ssize_t hnfsdev_read(struct file *file,	/* see include/linux/fs.h	*/
				char *buffer,	/* buffer to fill with data */
				size_t length,	/* length of the buffer	  */
				loff_t *offset /* "position in file", unused*/)
{
	/*
	 * Number of bytes actually written to the buffer
	 */
	int bytes_read = 0;

	/*
	 * Only allow one reading thread at a time.
	 */
	if (atomic_add_return(1, &in_devread) > 1) {
		atomic_dec(&in_devread);
		return -EBUSY;
	}

	while (length) {
		/* Test and / or initialize ll_current */
		INIT_LL_CURRENT;

		/* Here we go, write it out to user memory */
		switch (ll_current->pos) {
			case 0:
				put_user('2', buffer);	/* ll version */
				ll_current->pos++;
				break;
			case 1:
				put_user(ll_current->type, buffer);
				ll_current->pos++;
				break;
			case 2:
				put_user(ll_current->op, buffer);
				ll_current->pos++;
				break;
			default:
				if (ll_current->pos < ll_current->flen + 3) {
					put_user(ll_current->ino[ll_current->pos - 3], buffer);
					ll_current->pos++;
				} else {
					/* That's it, send newline and free the logline */
					char nl = '\n';
					put_user(nl, buffer);
					hnfsflt_ll_free(ll_current);
					ll_current = NULL;
				}
		}

		/* One byte more sent, adjust counters */
		bytes_read++;
		buffer++;
		length--;
	}

	atomic_dec(&in_devread);
	return bytes_read;
}


static unsigned int hnfsdev_poll(	struct file *file,
					struct poll_table_struct *wait)
{
	unsigned int pollval = 0;

	/* register wait before checking logline queue state */
	poll_wait(file, &read_waitq, wait);

	/*
	 * Only allow one accessing thread at a time for ll_current
	 */
	if (atomic_add_return(1, &in_devread) > 1) {
		atomic_dec(&in_devread);
		return -EBUSY;
	}
	do {
		int bytes_read = 1; /* pretend to have read something (will not block) */
		INIT_LL_CURRENT;
	} while (1 == 0);

	if (ll_current != NULL) {
		pollval = POLLIN | POLLRDNORM;
	}

	atomic_dec(&in_devread);

	return pollval;
}


/* Call-back functions for character device /dev/hnfs */
static struct file_operations fops = {
	.poll = hnfsdev_poll,
	.read = hnfsdev_read,
	.open = hnfsdev_open,
	.release = hnfsdev_release
};

int hnfsflt_dev_init(void)
{
	int major = register_chrdev(0, DEVICE_NAME, &fops);
	atomic_set(&hnfsflt_device_major, major);
	if (major < 0) {
		return major;
	}

	init_waitqueue_head(&read_waitq);

	printk(KERN_INFO "hnfsflt: logging char device has major number %d\n", major);
	return 0;
}

void hnfsflt_dev_cleanup(void)
{
	unregister_chrdev(atomic_read(&hnfsflt_device_major), DEVICE_NAME);
}
