#include "hnfsflt.h"

/* Ring buffer */
static struct hnfsflt_logline * rbuf[RBUF_SIZE];	/* ring buffer with pointers to log lines */
static int rbuf_head, rbuf_tail; /* Current positions in ring buffer */
static struct semaphore rbuf_full, rbuf_empty;	/* semaphores for producer-consumer synchronization */
#ifdef SPIN_LOCK_UNLOCKED
static spinlock_t rbuf_wlock = SPIN_LOCK_UNLOCKED;
#else
static DEFINE_SPINLOCK(rbuf_wlock);
#endif
atomic_t hnfsflt_noll = ATOMIC_INIT(0);  /* number of loglines in the ring buffer */

static unsigned long ll_status;	/* logline status */
#define LL_STATUS_DROPPED_SHIFT	0	/* bit 0x1: set when some loglines have been dropped */


/*
 * Set the logline dropped flag.
 */
static void set_ll_dropped(void) {
	set_bit(LL_STATUS_DROPPED_SHIFT, &ll_status);
}


/*
 * Test and clear the logline dropped flag.
 * return > 0 if there was some loglines have been dropped.
 */
int test_and_clear_ll_dropped(void) {
	return test_and_clear_bit(LL_STATUS_DROPPED_SHIFT, &ll_status);
}


/*
 * Allocates space for a log line.
 * The string path is copied (if non-NULL).
 */
struct hnfsflt_logline * hnfsflt_ll_alloc(
	const char type, const char op, const unsigned long ino, const unsigned long dev, const char *path)
{
	char ino_str[128];
	char dev_str[128];
	char *ll_ino = NULL, *ll_dev = NULL, *ll_path = NULL;
	struct hnfsflt_logline *logline = NULL;

	/* Copy path strings */
	size_t full_len;
	size_t ino_len;
	size_t dev_len;
	size_t path_len = 0;
	snprintf(ino_str, 128, "%lu", ino);
	ino_len = strlen(ino_str) + 1;
	full_len = ino_len;
	snprintf(dev_str, 128, "%lu", dev);
	dev_len = strlen(dev_str) + 1;
	full_len += dev_len;
	if (path) {
		path_len = strlen(path) + 1;
		full_len += path_len;
	}

	/* Allocate memory for strings and copy */
	ll_ino = kmalloc(full_len, GFP_KERNEL);
	if (!ll_ino)
		return ERR_PTR(-ENOMEM);

	ll_dev = ll_ino + ino_len;
	ll_path = ll_dev + dev_len;
	ll_dev[-1] = '\0';
	ll_path[-1] = '\0';
	ll_ino[full_len-1] = '\0';
	memcpy(ll_ino, ino_str, ino_len - 1);
	memcpy(ll_dev, dev_str, dev_len - 1);
	if (path)
		memcpy(ll_path, path, path_len - 1);

	/* Alloc memory for logline struct */
	logline = kmalloc(sizeof(struct hnfsflt_logline), GFP_KERNEL);
	if (!logline) {
		kfree(ll_ino);
		return ERR_PTR(-ENOMEM);
	}

	/* Initialize the logline */
	logline->ino = ll_ino;
	logline->dev = ll_dev;
	logline->path = ll_path;
	logline->type = type;
	logline->op = op;
	logline->pos = 0;
	logline->flen = full_len;

	return logline;
}


/*
 * Frees a logline if not NULL.
 * Doesn't care about the state of the sempahore!
 */
void hnfsflt_ll_free(struct hnfsflt_logline *logline) {
	if (logline) {
		kfree(logline->ino); /* frees ino, dev and path */
		kfree(logline);
	}
}


/*
 * Creates a logline and inserts it into ring buffer.
 * If the buffer is full, drop the oldest logline.
 * Returns 0 on success or a negative error value.
 */
void hnfsflt_ll_insert(struct hnfsflt_logline *logline)
{
	/* Insert into ringbuffer */
retry_rbuf_empty_lock:
	if (down_trylock(&rbuf_empty)) {
		struct hnfsflt_logline *loglineDropped = hnfsflt_ll_take(0);
		if (loglineDropped != NULL) {
			hnfsflt_ll_free(loglineDropped);
			set_ll_dropped();
		}
		goto retry_rbuf_empty_lock;
	}

	/* Remember that we have to read out this line */
	atomic_inc(&hnfsflt_noll);

	spin_lock(&rbuf_wlock);
	rbuf[rbuf_head] = logline;
	rbuf_head = (rbuf_head+1) % RBUF_SIZE;
	spin_unlock(&rbuf_wlock);

	up(&rbuf_full);
	wake_up(&read_waitq);
}


/*
 * Take the oldest logline.
 * return NULL if may_block is false and buffer empty
 * return ERR_PTR on error (interrupt)
 */
struct hnfsflt_logline *hnfsflt_ll_take(const int may_block)
{
	struct hnfsflt_logline *ll_oldest;

	if (down_trylock(&rbuf_full)) {
		/* If we have something, we give it back to the user.
			Otherwise we block. */
		if (!may_block)
			return NULL;

		if (down_interruptible(&rbuf_full)) {
			return ERR_PTR(-EINTR);
		}
	}

	ll_oldest = rbuf[rbuf_tail];
	rbuf_tail = (rbuf_tail+1) % RBUF_SIZE;

	/* Adjust counters */
	atomic_dec(&hnfsflt_noll);

	up(&rbuf_empty);

	return ll_oldest;
}


int hnfsflt_data_init(void)
{
	/* Initalize ring buffer semaphores */
	rbuf_head = rbuf_tail = 0;
	sema_init(&rbuf_full, 0);
	sema_init(&rbuf_empty, RBUF_SIZE);
	return 0;
}

void hnfsflt_data_cleanup(void)
{
	/* Purge rbuf */
	struct hnfsflt_logline *logline;
	while ((logline = hnfsflt_ll_take(0)) != NULL) {
		hnfsflt_ll_free(logline);
	}
}
